<?php
/**
 * Proof of Payment Image Generator
 * Generates a proof of payment image for Binance P2P orders
 */

/**
 * Generate proof of payment image
 * @param array $paymentData Payment details including:
 *   - date: Payment date
 *   - usdc_amount: Amount sent in USDC
 *   - eur_amount: Amount received in EUR
 *   - exchange_rate: Exchange rate used
 *   - bridge_transfer_id: Bridge transfer ID
 *   - receiver_name: Receiver name
 *   - iban: IBAN number
 *   - bank_name: Bank name
 *   - sepa_instant: Whether SEPA instant is supported (true/false)
 * @param string $outputPath Where to save the image
 * @return bool Success status
 */
function generateProofOfPaymentImage($paymentData, $outputPath) {
    // Image dimensions
    $width = 1200;
    $height = 1600;
    
    // Create image with white background
    $image = imagecreatetruecolor($width, $height);
    $white = imagecolorallocate($image, 255, 255, 255);
    $black = imagecolorallocate($image, 0, 0, 0);
    $gray = imagecolorallocate($image, 100, 100, 100);
    $darkGray = imagecolorallocate($image, 60, 60, 60);
    $blue = imagecolorallocate($image, 0, 102, 204);
    $lightGray = imagecolorallocate($image, 240, 240, 240);
    
    // Fill background
    imagefilledrectangle($image, 0, 0, $width, $height, $white);
    
    // Set up fonts (using built-in fonts, you can use TTF fonts if available)
    $fontLarge = 5;   // Largest built-in font
    $fontMedium = 4;
    $fontSmall = 3;
    $fontTiny = 2;
    
    // Y position tracking
    $y = 60;
    $lineHeight = 50;
    $sectionSpacing = 30;
    
    // Header: "Rampr Wallet"
    imagestring($image, $fontLarge, 50, $y, "Rampr Wallet", $blue);
    $y += 80;
    
    // Title: "Proof of Payment"
    imagestring($image, $fontMedium, 50, $y, "Proof of Payment", $black);
    $y += 60;
    
    // Draw separator line
    imageline($image, 50, $y, $width - 50, $y, $lightGray);
    $y += $sectionSpacing;
    
    // Payment Details Section
    imagestring($image, $fontSmall, 50, $y, "PAYMENT DETAILS", $darkGray);
    $y += 50;
    
    // Date
    $dateStr = "Date: " . ($paymentData['date'] ?? date('Y-m-d H:i:s'));
    imagestring($image, $fontSmall, 80, $y, $dateStr, $black);
    $y += $lineHeight;
    
    // Amount sent (USDC)
    $usdcStr = "Amount Sent (USDC): " . number_format($paymentData['usdc_amount'] ?? 0, 2) . " USDC";
    imagestring($image, $fontSmall, 80, $y, $usdcStr, $black);
    $y += $lineHeight;
    
    // Amount received (EUR)
    $eurStr = "Amount Received (EUR): " . number_format($paymentData['eur_amount'] ?? 0, 2) . " EUR";
    imagestring($image, $fontSmall, 80, $y, $eurStr, $black);
    $y += $lineHeight;
    
    // Exchange rate
    $rateStr = "Exchange Rate: " . ($paymentData['exchange_rate'] ?? 'N/A');
    imagestring($image, $fontSmall, 80, $y, $rateStr, $black);
    $y += $lineHeight;
    
    // Bridge Transfer ID
    $transferId = $paymentData['bridge_transfer_id'] ?? 'N/A';
    $transferIdStr = "Transfer ID: " . $transferId;
    imagestring($image, $fontSmall, 80, $y, $transferIdStr, $black);
    $y += $lineHeight + $sectionSpacing;
    
    // Draw separator
    imageline($image, 50, $y, $width - 50, $y, $lightGray);
    $y += $sectionSpacing;
    
    // Receiver Details Section
    imagestring($image, $fontSmall, 50, $y, "RECEIVER DETAILS", $darkGray);
    $y += 50;
    
    // Receiver Name
    $nameStr = "Receiver Name: " . ($paymentData['receiver_name'] ?? 'N/A');
    imagestring($image, $fontSmall, 80, $y, $nameStr, $black);
    $y += $lineHeight;
    
    // IBAN
    $iban = $paymentData['iban'] ?? 'N/A';
    $ibanStr = "IBAN: " . $iban;
    imagestring($image, $fontSmall, 80, $y, $ibanStr, $black);
    $y += $lineHeight;
    
    // Bank Name
    $bankStr = "Bank Name: " . ($paymentData['bank_name'] ?? 'N/A');
    imagestring($image, $fontSmall, 80, $y, $bankStr, $black);
    $y += $lineHeight + $sectionSpacing;
    
    // Draw separator
    imageline($image, 50, $y, $width - 50, $y, $lightGray);
    $y += $sectionSpacing;
    
    // Processing Time Information
    imagestring($image, $fontSmall, 50, $y, "PROCESSING TIME", $darkGray);
    $y += 50;
    
    // SEPA Instant payment timing info
    $sepaInstant = $paymentData['sepa_instant'] ?? false;
    if ($sepaInstant) {
        $timeInfo = "If receiver bank supports SEPA Instant payment, it may take up to 30 minutes. ";
    } else {
        $timeInfo = "If receiver bank does not support SEPA Instant payment, it may take up to 3 business days. ";
    }
    $timeInfo .= "Please check your bank account for confirmation.";
    
    // Word wrap text (basic implementation)
    $maxWidth = $width - 160;
    $words = explode(' ', $timeInfo);
    $line = '';
    foreach ($words as $word) {
        $testLine = $line . $word . ' ';
        // Rough estimate: each character is about 6 pixels wide
        if (strlen($testLine) * 6 > $maxWidth && $line !== '') {
            imagestring($image, $fontTiny, 80, $y, trim($line), $gray);
            $y += 35;
            $line = $word . ' ';
        } else {
            $line = $testLine;
        }
    }
    if (trim($line) !== '') {
        imagestring($image, $fontTiny, 80, $y, trim($line), $gray);
        $y += 35;
    }
    
    $y += $sectionSpacing;
    
    // Footer
    imagestring($image, $fontTiny, 50, $height - 40, "Generated by Rampr Wallet - " . date('Y-m-d H:i:s'), $gray);
    
    // Save image as JPEG
    $success = imagejpeg($image, $outputPath, 95);
    imagedestroy($image);
    
    return $success;
}

